import React, { useEffect, useState } from 'react';
import img from '@/assets/new-logo.png';
import './header.css';
import UserProfile from './UserProfile/UserProfile';
import MenuLinks from './MenuLinks/MenuLinks';
import { Link } from 'react-router-dom';
import AddList from './AddList/AddList';
import { Button } from '@/components/Elements/Button';
import { useAuth } from '@/lib/auth';
import storage from '@/utils/storage';
import { ConfirmDialogWithButton } from '@/components/Elements/ConfirmationDialog/ConfirmDialogWithButton';
import { usePersonStore } from '@/stores/currencyConverter';
import { useCurrencyListingUser } from '@/features/admin/api/getCurrencyListingUser';
import Box from '@mui/material/Box';
import Typography from '@mui/material/Typography';
import Modal from '@mui/material/Modal';
import i18next from 'i18next';

const style = {
  position: 'absolute' as const,
  top: '50%',
  left: '50%',
  transform: 'translate(-50%, -50%)',
  width: 400,
  bgcolor: 'background.paper',
  border: '2px solid #000',
  boxShadow: 24,
  p: 4,
};

const localCurrency = storage.getCurrency();

export enum HeaderType {
  LANDING = 'landing',
  DASHBOARD = 'dashboard',
  OURSERVICE = 'ourservice',
}

const loginHeader = localStorage.getItem('loginType');
const adminToken = localStorage.getItem('adminToken');

const Header = ({ type }: { type: HeaderType }) => {
  const [open, setOpen] = React.useState(false);
  const handleOpen = () => setOpen(true);
  const handleClose = () => setOpen(false);
  const { updateCurrency, setCurrencies } = usePersonStore();

  const lang = localStorage.getItem('lang') || 'no';

  const handleLangNorwegian = () => {
    localStorage.setItem('lang', 'no');
    window.location.reload();
  };

  const handleLangEnglish = () => {
    localStorage.setItem('lang', 'en');
    window.location.reload();
  };

  const { data } = useCurrencyListingUser({ page: '1' });

  const [curr, setCurr] = useState('');
  const handleCurr = (val: any) => {
    setCurr(val);
    handleOpen();
  };

  const [currencyData, setCurrencyData] = useState();

  useEffect(() => {
    if (data?.data) {
      setCurrencyData(data?.data);
      setCurrencies(data?.data);
    }
  }, [data]);

  const auth = useAuth();
  const handleSwithAdmin = () => {
    storage.clearToken();
    storage.setToken(adminToken);
    location.href = '/admin/property-requests';
    localStorage.removeItem('loginType');
    localStorage.removeItem('accessToken');
    localStorage.removeItem('adminToken');
  };

  const handleSwithAgency = () => {
    storage.clearToken();
    storage.setToken(adminToken);
    location.href = '/agency/property-requests';
    localStorage.removeItem('loginType');
    localStorage.removeItem('accessToken');
    localStorage.removeItem('adminToken');
  };
  const name = auth?.user?.first_name;
  const lastName = auth?.user?.last_name;

  return (
    <>
      <header>
        {loginHeader === 'admin' && (
          <div>
            <div className="AgentHeader centerAlignButton">
              <p>
                Login as{' '}
                <span className="htopNames">
                  {name} {lastName}
                </span>
                <ConfirmDialogWithButton
                  triggerButton={<button>Switch back to admin</button>}
                  confirmButton={
                    <Button onClick={handleSwithAdmin} variant="primary">
                      {i18next.t('Yes, Login')}
                    </Button>
                  }
                  title="Confirm"
                  icon="info"
                  body="Are you sure you want to Switch back to admin ?"
                />
              </p>
            </div>
          </div>
        )}
        {loginHeader === 'agency' && (
          <div>
            <div className="AgentHeader centerAlignButton">
              <p>
                Login as{' '}
                <span className="htopNames">
                  {name} {lastName}
                </span>
                <ConfirmDialogWithButton
                  triggerButton={<button>Switch back to agency</button>}
                  confirmButton={
                    <Button onClick={handleSwithAgency} variant="primary">
                      {i18next.t('Yes, Login')}
                    </Button>
                  }
                  title="Confirm"
                  icon="info"
                  body="Are you sure you want to Switch back to agency ?"
                />
              </p>
            </div>
          </div>
        )}
        <nav className="navbar navbar-expand-lg bg-dark navigation-main p-1">
          <div className="container">
            {' '}
            {currencyData && (
              <>
                <select
                  className="form-select currencyDrop"
                  aria-label="Default select example"
                  onChange={(e) => handleCurr(e.currentTarget.value)}
                  defaultValue={localCurrency}
                  value={localCurrency}
                >
                  <div>
                    {currencyData.map((item: any, index: any) => (
                      <option key={index} value={item.id}>
                        {item.symbol}
                      </option>
                    ))}
                  </div>
                </select>
              </>
            )}
          </div>
        </nav>
        <nav className="navbar navbar-expand-lg bg-light navigation-main">
          <div className="container header-navs">
            <Link className="navbar-brand" to="/">
              <img src={img} alt="logo" />
            </Link>
            {type === HeaderType.LANDING && <MenuLinks />}
            <div className="button-dash-content d-flex align-items-center ">
              {currencyData && (
                <>
                  <select
                    className="form-select currencyDrop"
                    aria-label="Default select example"
                    onChange={(e) => handleCurr(e.currentTarget.value)}
                    defaultValue={localCurrency}
                    value={localCurrency}
                  >
                    <div>
                      {currencyData.map((item: any, index: any) => (
                        <option key={index} value={item.id}>
                          {item.symbol}
                        </option>
                      ))}
                    </div>
                  </select>
                </>
              )}
              <AddList />
              <UserProfile />
              <div className="country-drop">
                <button
                  className={lang === 'no' ? 'lang-active' : 'lang-btn'}
                  onClick={handleLangNorwegian}
                  title="Norwegian"
                >
                  No
                </button>

                <button
                  className={lang === 'en' ? 'mx-3 lang-active' : 'mx-3 lang-btn'}
                  value="en"
                  onClick={handleLangEnglish}
                  title="English"
                >
                  En
                </button>
              </div>
            </div>
          </div>
        </nav>
      </header>

      <div>
        <Modal
          open={open}
          onClose={handleClose}
          aria-labelledby="modal-modal-title"
          aria-describedby="modal-modal-description"
        >
          <Box sx={style}>
            <Typography id="modal-modal-title" variant="h6" component="h2">
              <span className="modTitleSpan"> Confirmation</span>
            </Typography>
            <Typography id="modal-modal-description" sx={{ mt: 2 }}>
              Are you sure you want to change currency , will lead to refresh page data!
              <div className="row mt-4">
                <div className="col-md-6"></div>
                <div className="col-md-6 mkdCr">
                  <Button variant="outline" onClick={() => handleClose()}>
                    Cancel
                  </Button>

                  <Button onClick={() => updateCurrency(curr)}>Yes</Button>
                </div>
              </div>
            </Typography>
          </Box>
        </Modal>
      </div>
    </>
  );
};

export default Header;
